-- Video Earn Platform - Production database schema
-- All balances and financial logic are server-controlled.

-- Users
CREATE TABLE users (
  id                BIGSERIAL PRIMARY KEY,
  email             VARCHAR(255) NOT NULL UNIQUE,
  password_hash     VARCHAR(255) NOT NULL,
  email_verified_at TIMESTAMPTZ,
  balance_cents     BIGINT NOT NULL DEFAULT 0 CHECK (balance_cents >= 0),
  created_at        TIMESTAMPTZ NOT NULL DEFAULT NOW(),
  updated_at        TIMESTAMPTZ NOT NULL DEFAULT NOW(),
  suspended_at      TIMESTAMPTZ,
  risk_score        INT DEFAULT 0,
  last_active_at    TIMESTAMPTZ
);

CREATE INDEX idx_users_email ON users(email);
CREATE INDEX idx_users_verified ON users(email_verified_at);
CREATE INDEX idx_users_suspended ON users(suspended_at);
CREATE INDEX idx_users_last_active ON users(last_active_at);

-- Tasks (CPC stored server-side only; never exposed to frontend)
CREATE TABLE tasks (
  id              BIGSERIAL PRIMARY KEY,
  title           VARCHAR(500) NOT NULL,
  description     TEXT,
  duration_seconds INT NOT NULL DEFAULT 30,
  video_url       VARCHAR(500),  -- optional YouTube video link for watch task
  cpc_cents       INT NOT NULL,  -- CPC Pakistan, server-only
  active          BOOLEAN NOT NULL DEFAULT true,
  created_at      TIMESTAMPTZ NOT NULL DEFAULT NOW(),
  updated_at      TIMESTAMPTZ NOT NULL DEFAULT NOW()
);

CREATE INDEX idx_tasks_active ON tasks(active);
-- Migration if tasks already exists: ALTER TABLE tasks ADD COLUMN IF NOT EXISTS video_url VARCHAR(500);

-- User task completion (one completion per user per task; idempotent)
CREATE TABLE user_task_completion (
  id          BIGSERIAL PRIMARY KEY,
  user_id     BIGINT NOT NULL REFERENCES users(id),
  task_id     BIGINT NOT NULL REFERENCES tasks(id),
  reward_cents INT NOT NULL,  -- server-calculated: profit_pct * cpc
  completed_at TIMESTAMPTZ NOT NULL DEFAULT NOW(),
  UNIQUE(user_id, task_id)
);

CREATE INDEX idx_utc_user ON user_task_completion(user_id);
CREATE INDEX idx_utc_task ON user_task_completion(task_id);
CREATE INDEX idx_utc_completed_at ON user_task_completion(completed_at);

-- Withdrawals
CREATE TABLE withdrawals (
  id           BIGSERIAL PRIMARY KEY,
  user_id      BIGINT NOT NULL REFERENCES users(id),
  amount_cents BIGINT NOT NULL CHECK (amount_cents >= 1000),  -- min $10
  provider     VARCHAR(50) NOT NULL,  -- binance | okx
  status       VARCHAR(50) NOT NULL DEFAULT 'pending',  -- pending | approved | rejected | processing | completed | failed
  external_id  VARCHAR(255),  -- exchange tx id
  processed_at TIMESTAMPTZ,
  created_at   TIMESTAMPTZ NOT NULL DEFAULT NOW(),
  updated_at   TIMESTAMPTZ NOT NULL DEFAULT NOW()
);

CREATE INDEX idx_withdrawals_user ON withdrawals(user_id);
CREATE INDEX idx_withdrawals_status ON withdrawals(status);
CREATE INDEX idx_withdrawals_created ON withdrawals(created_at);

-- Admin settings (profit % 1-100; default 30)
CREATE TABLE admin_settings (
  id                BIGSERIAL PRIMARY KEY,
  key               VARCHAR(100) NOT NULL UNIQUE,
  value_json        TEXT NOT NULL,
  updated_at        TIMESTAMPTZ NOT NULL DEFAULT NOW(),
  updated_by_admin_id BIGINT
);

INSERT INTO admin_settings (key, value_json) VALUES ('profit_percentage', '30');

-- Fraud logs
CREATE TABLE fraud_logs (
  id          BIGSERIAL PRIMARY KEY,
  user_id     BIGINT REFERENCES users(id),
  event_type  VARCHAR(100) NOT NULL,
  payload_json TEXT,
  risk_score  INT,
  created_at  TIMESTAMPTZ NOT NULL DEFAULT NOW()
);

CREATE INDEX idx_fraud_logs_user ON fraud_logs(user_id);
CREATE INDEX idx_fraud_logs_created ON fraud_logs(created_at);
CREATE INDEX idx_fraud_logs_event ON fraud_logs(event_type);

-- Analytics aggregates (for dashboard/cache)
CREATE TABLE analytics_aggregates (
  id          BIGSERIAL PRIMARY KEY,
  metric_key  VARCHAR(100) NOT NULL,
  metric_value NUMERIC NOT NULL,
  period_from TIMESTAMPTZ NOT NULL,
  period_to   TIMESTAMPTZ NOT NULL,
  created_at  TIMESTAMPTZ NOT NULL DEFAULT NOW()
);

CREATE INDEX idx_analytics_key_period ON analytics_aggregates(metric_key, period_from, period_to);

-- KYC records
CREATE TABLE kyc_records (
  id            BIGSERIAL PRIMARY KEY,
  user_id       BIGINT NOT NULL REFERENCES users(id),
  tier          VARCHAR(50) NOT NULL DEFAULT 'basic',
  status        VARCHAR(50) NOT NULL DEFAULT 'pending',  -- pending | approved | rejected
  document_ref  VARCHAR(500),  -- encrypted storage reference
  reviewed_at   TIMESTAMPTZ,
  reviewed_by   BIGINT,
  created_at    TIMESTAMPTZ NOT NULL DEFAULT NOW(),
  updated_at    TIMESTAMPTZ NOT NULL DEFAULT NOW()
);

CREATE INDEX idx_kyc_user ON kyc_records(user_id);

-- User wallets (Binance / OKX addresses)
CREATE TABLE user_wallets (
  id        BIGSERIAL PRIMARY KEY,
  user_id   BIGINT NOT NULL REFERENCES users(id),
  provider  VARCHAR(50) NOT NULL,  -- binance | okx
  address   VARCHAR(500) NOT NULL,  -- stored encrypted at rest
  created_at TIMESTAMPTZ NOT NULL DEFAULT NOW(),
  UNIQUE(user_id, provider)
);

CREATE INDEX idx_user_wallets_user ON user_wallets(user_id);

-- Audit logs (immutable financial and admin actions)
CREATE TABLE audit_logs (
  id          BIGSERIAL PRIMARY KEY,
  actor_type  VARCHAR(50) NOT NULL,  -- user | admin | system
  actor_id    BIGINT,
  action      VARCHAR(100) NOT NULL,
  resource    VARCHAR(100),
  resource_id  BIGINT,
  payload_json TEXT,
  ip_address  INET,
  user_agent  TEXT,
  created_at  TIMESTAMPTZ NOT NULL DEFAULT NOW()
);

CREATE INDEX idx_audit_actor ON audit_logs(actor_type, actor_id);
CREATE INDEX idx_audit_created ON audit_logs(created_at);
CREATE INDEX idx_audit_resource ON audit_logs(resource, resource_id);
