# Production Deployment Architecture

## Web server: NGINX

- **HTTPS enforcement**: Redirect HTTP → HTTPS (301). Listen 443 with SSL certificate (e.g. Let’s Encrypt).
- **HSTS**: Add header `Strict-Transport-Security: max-age=31536000; includeSubDomains; preload`.
- **CSP headers**: Content-Security-Policy to restrict script sources, inline scripts (avoid if possible), and connections to your API origin only.
- **Secure cookies**: Session cookie with `Secure`, `HttpOnly`, `SameSite=Strict` (or Lax if cross-origin needed).
- **Rate limiting**: `limit_req_zone` and `limit_req` for IP-based request limits; stricter for /api/auth and /api/tasks/complete, /api/withdrawals.
- **Static assets**: Serve HTML, CSS, JS, manifest from NGINX; cache control for immutable assets.

Example snippet (tune to your paths and API backend):

```nginx
# Rate limit
limit_req_zone $binary_remote_addr zone=api:10m rate=10r/s;
limit_req_zone $binary_remote_addr zone=auth:10m rate=3r/m;

server {
  listen 443 ssl http2;
  ssl_certificate /path/to/fullchain.pem;
  ssl_certificate_key /path/to/privkey.pem;
  add_header Strict-Transport-Security "max-age=31536000; includeSubDomains; preload" always;
  add_header Content-Security-Policy "default-src 'self'; script-src 'self'; style-src 'self' 'unsafe-inline';" always;

  location /api/ {
    limit_req zone=api burst=20 nodelay;
    proxy_pass http://backend;
    proxy_set_header Host $host;
    proxy_set_header X-Real-IP $remote_addr;
    proxy_set_header X-Forwarded-For $proxy_add_x_forwarded_for;
    proxy_set_header X-Forwarded-Proto $scheme;
  }
  location /api/auth/ {
    limit_req zone=auth burst=5 nodelay;
    proxy_pass http://backend;
    # ... same proxy headers
  }
  location / {
    root /var/www/video-earn;
    try_files $uri $uri/ /index.html;
  }
}
```

## CSRF protection

- Use same-site cookies and/or CSRF tokens for state-changing API requests (POST/PUT/DELETE).
- Validate token server-side on each mutation; reject if missing or invalid.

## Database

- **Encryption**: Enable TDE or use a DB offering encryption at rest.
- **Access**: DB not publicly reachable; only app and backup services in private network.
- **Backups**: Daily automated backups; retain per retention policy; test restore periodically.
- **Credentials**: From env or secrets manager; never in repo.

## Logging and monitoring

- **Log aggregation**: Send app and access logs to centralized logging (e.g. ELK, Loki, CloudWatch).
- **SIEM**: Forward security-relevant events (auth failures, fraud_logs, audit_logs, admin actions) to SIEM for alerting and correlation.
- **Alerts**: Failed logins spike, high fraud score, withdrawal failures, API errors above threshold.

## Cloud deployment (AWS / GCP / Azure)

- **Compute**: Run backend and workers in private subnet; load balancer in front; auto-scaling group by CPU/request count.
- **Database**: Managed DB (RDS, Cloud SQL, etc.) in private subnet; multi-AZ for HA.
- **Secrets**: Use AWS Secrets Manager / GCP Secret Manager / Azure Key Vault for DB and API keys.
- **CDN**: Serve static assets (HTML, CSS, JS) via CDN (CloudFront, Cloud CDN, etc.) with cache invalidation on deploy.
- **DDoS mitigation**: Use cloud provider DDoS protection (e.g. AWS Shield, Cloud Armor) and rate limiting at edge and NGINX.

## Auto-scaling

- Scale app tier by request rate or CPU; scale workers by queue depth.
- Define min/max instances and scaling policies; use health checks for LB.

## Security checklist

- No inline secrets or business logic exposure in frontend.
- All financial calculations and CPC/profit % handling server-side only.
- Strict role separation; admin APIs require admin role.
- Financial and audit logs immutable (append-only; no user delete).
- Assume adversarial environment: validate and authorize every request server-side.
